const express = require('express');
const fs = require('fs');
const path = require('path');
const { spawn } = require('child_process');

const app = express();

// Porta do servidor do "aprender" (independente do server.js da raiz)
const PORT = parseInt(process.env.PORT || '3100', 10);

// URL do tradutor.
// Preferência (sem Docker): Argos Translate (Python) rodando localmente.
//   - default: http://127.0.0.1:5100/translate
// Opcional (com Docker): LibreTranslate em http://localhost:5000/translate
const ARGOS_URL = String(process.env.ARGOS_URL || 'http://127.0.0.1:5100/translate');
const LIBRE_URL = String(process.env.LIBRE_URL || 'http://localhost:5000/translate');
const PIVOT_LANG = String(process.env.PIVOT_LANG || 'en').toLowerCase();

// Python usado pelo servidor para instalar modelos Argos sob demanda.
// Por padrão usa "python" do PATH.
// Se existir um venv local em .venv, preferimos ele automaticamente.
// Você também pode forçar via variável de ambiente:
//   - Windows: $env:PYTHON_EXE = "C:\\...\\.venv\\Scripts\\python.exe"
//   - Linux/mac: export PYTHON_EXE="/path/.venv/bin/python"
const DEFAULT_VENV_PY_WIN = path.join(__dirname, '.venv', 'Scripts', 'python.exe');
const DEFAULT_VENV_PY_NIX = path.join(__dirname, '.venv', 'bin', 'python');
const PYTHON_EXE = String(
  process.env.PYTHON_EXE ||
    (fs.existsSync(DEFAULT_VENV_PY_WIN)
      ? DEFAULT_VENV_PY_WIN
      : fs.existsSync(DEFAULT_VENV_PY_NIX)
        ? DEFAULT_VENV_PY_NIX
        : 'python')
);

// Por que “traduzir” pode demorar?
// 1) Primeira vez (cold start):
//    - Node ainda está “aquecendo”
//    - Python/Argos pode estar carregando modelos em cache
// 2) Primeiro uso de um par novo (ex.: pt->de):
//    - o servidor tenta baixar e instalar o(s) modelo(s) do Argos (internet + download)
//    - isso pode levar dezenas de segundos/minutos
// 3) Quando não existe modelo Argos:
//    - cai em fallback (LibreTranslate local ou serviços públicos), que pode ser lento

// Pasta que será servida como estática: aprender
const APRENDER_DIR = path.resolve(__dirname, '..');

app.use((req, res, next) => {
  // Mesmas headers usadas no server.js da raiz (útil para recursos com workers/wasm)
  res.setHeader('Cross-Origin-Opener-Policy', 'same-origin');
  res.setHeader('Cross-Origin-Embedder-Policy', 'require-corp');
  next();
});

app.use(express.json({ limit: '5mb' }));
app.use(express.static(APRENDER_DIR));

// --- Argos auto-install (sem Docker) ---------------------------------------
// Quando o usuário escolhe um idioma novo, pode não existir modelo instalado.
// Aqui fazemos instalação SOB DEMANDA (primeira vez que aquele par é usado).
// Requer:
// - Python instalado
// - requirements.txt instalado (argostranslate)
// - internet para baixar os modelos

let argosInstallLock = Promise.resolve();

// Cache simples: evita chamar /health do Argos a cada request.
// Isso reduz custo quando você instala muitos modelos (listar todos pode ficar caro).
let argosInstalledPairsCache = null;
let argosInstalledPairsCacheAtMs = 0;

function nowMs() {
  return Date.now();
}

function argosHealthUrl() {
  // ARGOS_URL: http://127.0.0.1:5100/translate
  return String(ARGOS_URL).replace(/\/translate\/?$/, '/health');
}

async function fetchJson(url, options) {
  const r = await fetch(url, options);
  const text = await r.text().catch(() => '');
  let json = null;
  try {
    json = text ? JSON.parse(text) : null;
  } catch (_) {
    json = null;
  }
  return { ok: r.ok, status: r.status, text, json };
}

async function getArgosInstalledPairs() {
  const url = argosHealthUrl();
  const { ok, status, json, text } = await fetchJson(url, { method: 'GET' });
  if (!ok) throw new Error(`Argos /health falhou (HTTP ${status}). ${text || ''}`.trim());
  const installed = Array.isArray(json?.installed) ? json.installed : [];
  return new Set(installed.map((p) => `${String(p.from).toLowerCase()}->${String(p.to).toLowerCase()}`));
}

async function getArgosInstalledPairsCached(ttlMs = 15000) {
  const age = nowMs() - argosInstalledPairsCacheAtMs;
  if (argosInstalledPairsCache && age >= 0 && age <= ttlMs) return argosInstalledPairsCache;
  const pairs = await getArgosInstalledPairs();
  argosInstalledPairsCache = pairs;
  argosInstalledPairsCacheAtMs = nowMs();
  return pairs;
}

function runPythonInstall(from, to) {
  return new Promise((resolve, reject) => {
    const scriptPath = path.join(__dirname, 'argos_setup.py');
    const args = [scriptPath, '--from', String(from), '--to', String(to), '--json'];

    const child = spawn(PYTHON_EXE, args, { cwd: __dirname, windowsHide: true });
    let out = '';
    let err = '';
    child.stdout.on('data', (d) => (out += String(d)));
    child.stderr.on('data', (d) => (err += String(d)));
    child.on('error', (e) => reject(e));
    child.on('close', (code) => {
      let parsed = null;
      try {
        parsed = out.trim() ? JSON.parse(out.trim().split(/\r?\n/).pop()) : null;
      } catch (_) {
        parsed = null;
      }
      resolve({ code: Number(code), out: out.trim(), err: err.trim(), json: parsed });
    });
  });
}

function pairsNeededForPivot(source, target, pivot) {
  const s = String(source).toLowerCase();
  const t = String(target).toLowerCase();
  const p = String(pivot).toLowerCase();
  if (!s || !t || s === t) return [];
  const pairs = [];
  if (s !== p) pairs.push([s, p]);
  if (t !== p) pairs.push([p, t]);
  return pairs;
}

async function ensureArgosModels(source, target) {
  // Serializa instalações para evitar corrida entre requests.
  argosInstallLock = argosInstallLock.then(async () => {
    const installed = await getArgosInstalledPairsCached();
    const needed = pairsNeededForPivot(source, target, PIVOT_LANG);

    for (const [from, to] of needed) {
      const key = `${from}->${to}`;
      if (installed.has(key)) continue;

      const r = await runPythonInstall(from, to);
      if (r.code === 0) {
        installed.add(key);

        // Atualiza o cache local para não precisar refazer /health no próximo request.
        argosInstalledPairsCache = installed;
        argosInstalledPairsCacheAtMs = nowMs();
        continue;
      }

      // code 2: par não existe no índice
      const hint = r.err || r.out || '';
      const msg = `Não existe modelo Argos para ${from}->${to} (pivot ${PIVOT_LANG}). ${hint}`.trim();
      const e = new Error(msg);
      e.name = 'ArgosModelUnavailable';
      throw e;
    }
  });

  return argosInstallLock;
}

function setCors(req, res) {
  const origin = req.headers.origin ? String(req.headers.origin) : '*';
  res.setHeader('Access-Control-Allow-Origin', origin);
  res.setHeader('Vary', 'Origin');
  res.setHeader('Access-Control-Allow-Methods', 'POST, OPTIONS');
  res.setHeader('Access-Control-Allow-Headers', 'Content-Type');
}

app.get('/health', (req, res) => {
  res.json({ ok: true, port: PORT, argosUrl: ARGOS_URL, libreUrl: LIBRE_URL });
});

app.options('/api/translate', (req, res) => {
  setCors(req, res);
  res.sendStatus(204);
});

// POST /api/translate
// Body: { text, source, target }
// - source/target: códigos base (pt, fr, en, it...)
// - chama o LibreTranslate (ou outra API compatível) configurada por TRANSLATE_URL
app.post('/api/translate', async (req, res) => {
  try {
    setCors(req, res);

    // Telemetria simples (didática): mede onde está gastando tempo.
    // Isso ajuda a entender “porque demorou”:
    // - ensureModelsMs: tempo instalando/verificando modelos
    // - attemptMs: tempo de cada backend
    const t0 = nowMs();
    const timingsMs = { ensureModelsMs: 0, attempts: [] };

    const text = typeof req.body?.text === 'string' ? req.body.text.trim() : '';
    const source = typeof req.body?.source === 'string' ? req.body.source.trim() : '';
    const target = typeof req.body?.target === 'string' ? req.body.target.trim() : '';

    if (!text) return res.status(400).json({ message: 'Campo "text" vazio.' });
    if (!source) return res.status(400).json({ message: 'Campo "source" vazio.' });
    if (!target) return res.status(400).json({ message: 'Campo "target" vazio.' });

    // pt-BR e pt-PT (no front) viram ambos "pt". Evita erro: traduzir pt->pt.
    if (String(source).toLowerCase() === String(target).toLowerCase()) {
      return res.json({ translatedText: text, noOp: true });
    }

    // Payload compatível com Argos service e LibreTranslate.
    const payload = { q: text, source, target, format: 'text' };

    // 1) Antes de tentar Argos, garante que os modelos necessários existem.
    // Isso deixa "todos os idiomas" funcionando conforme você vai usando.
    const tEnsure0 = nowMs();
    try {
      await ensureArgosModels(source, target);
      timingsMs.ensureModelsMs = nowMs() - tEnsure0;
    } catch (e) {
      // Se não existe modelo para algum par, seguimos para fallback (Libre/público).
      // Se o Argos não estiver rodando, também vamos cair no fallback.
      // (O erro detalhado será retornado se TODOS os fallbacks falharem.)
      timingsMs.ensureModelsMs = nowMs() - tEnsure0;
    }

    const urls = [
      // 1) Argos local (sem Docker)
      ARGOS_URL,
      // 2) LibreTranslate local (Docker) (se existir)
      LIBRE_URL,
      // 3) fallback público (pode falhar/limitar)
      'https://translate.argosopentech.com/translate',
      'https://libretranslate.de/translate'
    ].filter(Boolean);

    /** @type {{url?: string, status?: number, detail?: string, message?: string}} */
    let lastErr = {};

    for (const url of urls) {
      let r;
      const tAttempt0 = nowMs();
      try {
        r = await fetch(url, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify(payload)
        });
      } catch (e) {
        timingsMs.attempts.push({ url, ms: nowMs() - tAttempt0, ok: false, status: 0 });
        lastErr = {
          url,
          status: 0,
          message: `Falha ao conectar no tradutor em ${url}.`,
          detail: e && e.message ? String(e.message) : String(e)
        };
        continue;
      }

      if (!r.ok) {
        const bodyText = await r.text().catch(() => '');
        timingsMs.attempts.push({ url, ms: nowMs() - tAttempt0, ok: false, status: r.status });
        lastErr = {
          url,
          status: r.status,
          message: `Tradutor retornou HTTP ${r.status} em ${url}.`,
          detail: bodyText
        };
        continue;
      }

      const data = await r.json().catch(() => null);
      const translatedText = data && data.translatedText ? String(data.translatedText) : '';
      if (!translatedText.trim()) {
        timingsMs.attempts.push({ url, ms: nowMs() - tAttempt0, ok: false, status: 502 });
        lastErr = {
          url,
          status: 502,
          message: `Tradutor retornou vazio em ${url}.`,
          detail: JSON.stringify(data || {})
        };
        continue;
      }

      timingsMs.attempts.push({ url, ms: nowMs() - tAttempt0, ok: true, status: 200 });
      timingsMs.totalMs = nowMs() - t0;
      return res.json({ translatedText, backend: url, timingsMs });
    }

    // Mensagem mais direcionada quando Argos local não está rodando.
    if (String(lastErr.url || '').includes('5100')) {
      lastErr.detail =
        (lastErr.detail ? `${lastErr.detail}\n` : '') +
        'Dica: rode o Argos local: python argos_service.py (na pasta translate_server).';
    }

    // Se não conseguimos instalar modelo, diga claramente o motivo.
    if (String(lastErr.detail || '').includes('Não existe modelo Argos')) {
      lastErr.detail +=
        `\nDica: nem todos os idiomas do mundo têm modelo Argos. Para esses casos, use LibreTranslate (Docker) ou uma API paga.`;
    }

    return res.status(502).json({
      message: lastErr.message || 'Falha na tradução.',
      detail: lastErr.detail || '',
      backend: lastErr.url || '',
      timingsMs
    });
  } catch (e) {
    return res.status(500).json({
      message: `Erro interno na tradução: ${e && e.message ? e.message : String(e)}`
    });
  }
});

app.listen(PORT, () => {
  console.log(`aprender server: http://localhost:${PORT}/microfonestt.html`);
  console.log(`Argos backend (python): ${ARGOS_URL}`);
  console.log(`Libre backend (docker): ${LIBRE_URL}`);
});
