import argparse
import json

import argostranslate.package


def install_pair(from_code: str, to_code: str) -> bool:
    # Normaliza os códigos (Argos usa códigos base: pt, en, fr, es...)
    from_code = from_code.lower().strip()
    to_code = to_code.lower().strip()

    # 1) Se já estiver instalado, não faz nada.
    # Isso deixa o "instalar sob demanda" rápido quando o par já existe.
    for pkg in argostranslate.package.get_installed_packages():
        try:
            if pkg.from_code == from_code and pkg.to_code == to_code:
                return True
        except Exception:
            continue

    # 2) Atualiza o índice remoto de pacotes disponíveis.
    # IMPORTANTE: esta etapa pode demorar, porque faz acesso à internet.
    # Sem isso, o Argos não sabe quais modelos existem para baixar.
    argostranslate.package.update_package_index()
    available = argostranslate.package.get_available_packages()

    match = None
    for pkg in available:
        if pkg.from_code == from_code and pkg.to_code == to_code:
            match = pkg
            break

    if not match:
        # Nem todo par existe no Argos.
        # Ex.: alguns idiomas do seu combobox podem não ter modelo.
        print(f"ERRO: não achei pacote para {from_code}->{to_code} no índice.")
        return False

    # 3) Baixa o arquivo do modelo.
    # Esta é a parte MAIS lenta quando é o primeiro uso daquele idioma,
    # porque envolve download (e depende da sua conexão).
    path = match.download()

    # 4) Instala o modelo baixado no ambiente Python.
    # Depois disso, o serviço `argos_service.py` passa a conseguir traduzir.
    argostranslate.package.install_from_path(path)
    print(f"OK: instalado {from_code}->{to_code}")
    return True


def main():
    # Este script é chamado pelo servidor Node (server.js) quando você usa um idioma
    # pela primeira vez. Ele permite instalar modelos automaticamente.
    p = argparse.ArgumentParser(description="Instala modelos Argos Translate")
    p.add_argument("--from", dest="from_code", required=True, help="Idioma origem (ex: pt)")
    p.add_argument("--to", dest="to_code", required=True, help="Idioma destino (ex: it)")
    p.add_argument("--json", action="store_true", help="Imprime resultado em JSON")
    args = p.parse_args()

    ok = install_pair(args.from_code, args.to_code)
    if args.json:
        print(json.dumps({"ok": bool(ok), "from": args.from_code, "to": args.to_code}, ensure_ascii=False))

    # Exit code para integração:
    # 0: instalado/ok
    # 2: não existe pacote
    raise SystemExit(0 if ok else 2)


if __name__ == "__main__":
    main()
