import json
import os
from http.server import BaseHTTPRequestHandler, HTTPServer

import argostranslate.package
import argostranslate.translate


# Guarda uma “assinatura” do conjunto de modelos instalados.
# Se mudar, limpamos o cache do Argos na próxima request.
_LAST_INSTALLED_SIGNATURE = None


# Por que pode “demorar” aqui?
# - Tradução direta (source->target) costuma ser rápida.
# - Pivot (source->en->target) faz DUAS traduções, então é naturalmente mais lenta.
# - Quando novos modelos são instalados com o serviço já rodando, o Argos mantém
#   cache interno (lru_cache) da lista de idiomas instalados.
#   Para enxergar modelos recém-instalados, precisamos limpar esse cache.


def _read_json(handler: BaseHTTPRequestHandler):
    length = int(handler.headers.get("Content-Length", "0") or "0")
    if length <= 0:
        return {}
    raw = handler.rfile.read(length)
    try:
        return json.loads(raw.decode("utf-8"))
    except Exception:
        return {}


def _write_json(handler: BaseHTTPRequestHandler, status: int, payload: dict, add_headers=None):
    body = json.dumps(payload, ensure_ascii=False).encode("utf-8")
    handler.send_response(status)
    if callable(add_headers):
        add_headers()
    handler.send_header("Content-Type", "application/json; charset=utf-8")
    handler.send_header("Content-Length", str(len(body)))
    handler.end_headers()
    handler.wfile.write(body)


def _has_installed_pair(source: str, target: str) -> bool:
    installed = argostranslate.package.get_installed_packages()
    for pkg in installed:
        try:
            if pkg.from_code == source and pkg.to_code == target:
                return True
        except Exception:
            continue
    return False


def _translate_with_pivot(text: str, source: str, target: str, pivot: str) -> str:
    # 1) source -> pivot
    intermediate = argostranslate.translate.translate(text, source, pivot)
    # 2) pivot -> target
    return argostranslate.translate.translate(intermediate, pivot, target)


class Handler(BaseHTTPRequestHandler):
    def _cors(self):
        origin = self.headers.get("Origin") or "*"
        self.send_header("Access-Control-Allow-Origin", origin)
        self.send_header("Vary", "Origin")
        self.send_header("Access-Control-Allow-Methods", "POST, OPTIONS, GET")
        self.send_header("Access-Control-Allow-Headers", "Content-Type")

    def do_OPTIONS(self):
        self.send_response(204)
        self._cors()
        self.end_headers()

    def do_GET(self):
        if self.path == "/health":
            installed = []
            for pkg in argostranslate.package.get_installed_packages():
                installed.append({"from": pkg.from_code, "to": pkg.to_code})
            _write_json(self, 200, {"ok": True, "installed": installed}, self._cors)
            return

        _write_json(self, 404, {"message": "Not found"}, self._cors)

    def do_POST(self):
        if self.path not in ("/translate", "/api/translate"):
            _write_json(self, 404, {"message": "Not found"}, self._cors)
            return

        data = _read_json(self)
        text = (data.get("q") or data.get("text") or "").strip()
        source = (data.get("source") or "").strip().lower()
        target = (data.get("target") or "").strip().lower()

        if not text:
            _write_json(self, 400, {"message": "Campo 'text'/'q' vazio."}, self._cors)
            return
        if not source:
            _write_json(self, 400, {"message": "Campo 'source' vazio."}, self._cors)
            return
        if not target:
            _write_json(self, 400, {"message": "Campo 'target' vazio."}, self._cors)
            return

        # Caso sem tradução (ex.: pt->pt). Evita erro de modelo inexistente.
        if source == target:
            _write_json(self, 200, {"translatedText": text}, self._cors)
            return

        # Cache do Argos (importante para “instalação sob demanda”):
        # O Argos usa lru_cache em get_installed_languages(). Se você instala um modelo
        # NOVO com o serviço já rodando, o processo pode não enxergar o novo modelo.
        #
        # Para NÃO prejudicar performance, a gente só limpa o cache quando detecta que
        # o conjunto de pacotes instalados mudou.
        global _LAST_INSTALLED_SIGNATURE
        try:
            pkgs = argostranslate.package.get_installed_packages()
            sig = '|'.join(sorted([f"{p.from_code}->{p.to_code}" for p in pkgs]))
        except Exception:
            sig = ''

        if sig != _LAST_INSTALLED_SIGNATURE:
            try:
                argostranslate.translate.get_installed_languages.cache_clear()
            except Exception:
                pass
            _LAST_INSTALLED_SIGNATURE = sig

        pivot = os.environ.get("ARGOS_PIVOT", "en").strip().lower() or "en"
        allow_pivot = os.environ.get("ARGOS_ALLOW_PIVOT", "1").strip() not in ("0", "false", "False")

        try:
            if _has_installed_pair(source, target):
                translated = argostranslate.translate.translate(text, source, target)
            else:
                if not allow_pivot:
                    _write_json(
                        self,
                        400,
                        {
                            "message": f"Modelo Argos não instalado para {source}->{target}.",
                            "detail": "Instale o modelo direto com: python argos_setup.py --from <src> --to <dst>"
                        },
                        self._cors,
                    )
                    return

                # Pivot padrão (normalmente EN) quando não existe par direto.
                if not (_has_installed_pair(source, pivot) and _has_installed_pair(pivot, target)):
                    _write_json(
                        self,
                        400,
                        {
                            "message": f"Sem modelo direto {source}->{target} e sem modelos para pivot via {pivot}.",
                            "detail": (
                                f"Instale: python argos_setup.py --from {source} --to {pivot}  e  "
                                f"python argos_setup.py --from {pivot} --to {target}"
                            )
                        },
                        self._cors,
                    )
                    return

                translated = _translate_with_pivot(text, source, target, pivot)
        except Exception as e:
            _write_json(self, 500, {"message": "Falha ao traduzir (Argos).", "detail": str(e)}, self._cors)
            return

        _write_json(self, 200, {"translatedText": translated}, self._cors)

    def log_message(self, format: str, *args):
        # Silencia logs chatos no console
        return


def main():
    port = int(os.environ.get("ARGOS_PORT", "5100"))
    host = os.environ.get("ARGOS_HOST", "127.0.0.1")

    httpd = HTTPServer((host, port), Handler)
    print(f"Argos local translator: http://{host}:{port}/translate")
    print("Health: /health")
    httpd.serve_forever()


if __name__ == "__main__":
    main()
