// Modo STT (Speech-to-Text) usando Web Speech API.
// Importante: aqui NÃO usamos getUserMedia(); o próprio SpeechRecognition pede permissão do microfone.

// IIFE (função auto-executável) para não “sujar” o escopo global com variáveis internas.
(() => {
	// Pega o botão pelo id no HTML (pode ser null se o id não existir).
	/** @type {HTMLButtonElement|null} */
	const btnSTT = document.getElementById('btnSTT');

	// Pega o elemento de status pelo id no HTML.
	/** @type {HTMLElement|null} */
	const statusEl = document.getElementById('status');

	// Pega o elemento de saída (onde vamos listar transcrições finais).
	/** @type {HTMLElement|null} */
	const saidaEl = document.getElementById('saida');

	// Botão para o sistema falar (TTS) o que foi reconhecido.
	/** @type {HTMLButtonElement|null} */
	const btnFalar = document.getElementById('btnFalar');

	// Botão para limpar o painel de saída (transcrições e comparações).
	/** @type {HTMLButtonElement|null} */
	const btnLimparSaida = document.getElementById('btnLimparSaida');

	// Campo (edit) onde o usuário digita a palavra/frase esperada.
	/** @type {HTMLInputElement|null} */
	const txtEsperado = document.getElementById('txtEsperado');

	// Botão que compara o texto esperado com a última fala reconhecida.
	/** @type {HTMLButtonElement|null} */
	const btnComparar = document.getElementById('btnComparar');

	// Botão "Traduzir": traduz o texto do input e fala (TTS) no idioma destino.
	/** @type {HTMLButtonElement|null} */
	const btnTraduzir = document.getElementById('btnTraduzir');

	// Wrapper + campo (edit) para mostrar a frase já traduzida.
	/** @type {HTMLElement|null} */
	const wrapTraduzido = document.getElementById('wrapTraduzido');
	/** @type {HTMLInputElement|null} */
	const txtTraduzido = document.getElementById('txtTraduzido');

	// Pega o select (combobox) do idioma.
	/** @type {HTMLSelectElement|null} */
	const selLang = document.getElementById('selLang');

	// Segundo select: idioma NATIVO (origem) do usuário (usado para tradução).
	/** @type {HTMLSelectElement|null} */
	const selLangNativo = document.getElementById('selLangNativo');

	// Pega o select (combobox) do tempo (segundos) para auto-desligar o STT.
	/** @type {HTMLSelectElement|null} */
	const selTempo = document.getElementById('selTempo');

	// Combobox para controlar a velocidade do TTS (SpeechSynthesisUtterance.rate).
	/** @type {HTMLSelectElement|null} */
	const selTtsRate = document.getElementById('selTtsRate');

	// Elemento onde mostramos o contador regressivo (quando o tempo não é null).
	/** @type {HTMLElement|null} */
	const contadorEl = document.getElementById('contador');

	// Descobre a implementação de SpeechRecognition (padrão ou prefixada no Chrome antigo).
	const SpeechRecognitionImpl = window.SpeechRecognition || window.webkitSpeechRecognition;

	// Guarda a instância do reconhecimento (inicialmente não criada).
	/** @type {SpeechRecognition|null} */
	let recognition = null;

	// Estado simples para a UI saber se está ativo.
	let sttAtivo = false;

	// Guarda a última transcrição FINAL (para o botão "Falar").
	let ultimaTranscricaoFinal = '';

	// ===== Idiomas (carregados de langspri.js) =====
	// O arquivo `langspri.js` define duas estruturas globais:
	// - window.LANGS:  { fr: "Francês", ... }
	// - window.LANGS_BCP: { fr: "fr-FR", ... }
	// Aqui, usamos isso para popular o combobox #selLang automaticamente.
	// Assim você mantém a lista de idiomas em um único lugar.
	function popularIdiomasNoCombobox(selectEl, preferidoBcp47) {
		// Esta função preenche um <select> com as opções de idioma.
		// Ela é usada em DOIS lugares:
		// - `#selLang` (idioma destino: STT/TTS)
		// - `#selLangNativo` (idioma nativo/origem: tradução)
		if (!selectEl) return;

		// Lê as listas globais (se existirem).
		const nomes = window.LANGS || null;
		const bcps = window.LANGS_BCP || null;

		// Se não temos as listas globais, usamos um fallback mínimo (não quebra a página).
		// Isso também ajuda se alguém abrir este HTML sem carregar langspri.js.
		const fallback = [
			{ label: 'Português (BR)', value: 'pt-BR' },
			{ label: 'Français (France)', value: 'fr-FR' }
		];

		/** @type {{label: string, value: string}[]} */
		let itens = [];

		if (nomes && bcps) {
			// Monta a lista a partir das chaves (ex.: fr, en, ptbr...).
			// Observação: a ordem costuma seguir a ordem de inserção do objeto no JS.
			for (const key of Object.keys(bcps)) {
				const nome = nomes[key] ? String(nomes[key]) : String(key);
				const bcp = String(bcps[key] || '').trim();
				if (!bcp) continue;
				itens.push({ label: `${nome} - ${bcp}`, value: bcp });
			}
		} else {
			itens = fallback.map((x) => ({ label: `${x.label} - ${x.value}`, value: x.value }));
		}

		// Se por algum motivo não gerou itens, usa fallback.
		if (!itens.length) {
			itens = fallback.map((x) => ({ label: `${x.label} - ${x.value}`, value: x.value }));
		}

		// Guarda o valor atual (se houver) para tentar preservar a seleção.
		const valorAtual = String(selectEl.value || '').trim();

		// Limpa todas as opções atuais e recria.
		selectEl.innerHTML = '';
		for (const item of itens) {
			const opt = document.createElement('option');
			opt.value = item.value;
			opt.textContent = item.label;
			selectEl.appendChild(opt);
		}

		// Define o valor selecionado:
		// 1) mantém o que já estava (se existir na lista)
		// 2) senão, prefere pt-BR (se existir)
		// 3) senão, cai na primeira opção
		if (valorAtual && itens.some((x) => x.value === valorAtual)) {
			selectEl.value = valorAtual;
			return;
		}

		// Se não existe seleção anterior, tenta aplicar um idioma preferido (se existir no select).
		if (preferidoBcp47 && itens.some((x) => x.value === preferidoBcp47)) {
			selectEl.value = preferidoBcp47;
			return;
		}

		// Se não há preferido válido, cai na primeira opção.
		selectEl.selectedIndex = 0;
	}

	// ===== Temporizador do STT (auto-desligar) =====
	// Quando o usuário escolhe um tempo (ex.: 5s), o STT liga e desliga sozinho.
	let sttAutoStopTimeoutId = null;
	let sttCountdownIntervalId = null;
	let sttAutoStopFimMs = 0;

	// Guarda o tempo selecionado no momento do start() (para iniciar no onstart).
	let sttTempoSolicitadoSegundos = null;

	// Atualiza o texto do contador na tela.
	function setContador(texto) {
		if (!contadorEl) return;
		contadorEl.textContent = String(texto || '');
	}

	// Lê o tempo (segundos) escolhido no combobox.
	// Retorno:
	// - null: não auto-desliga
	// - number > 0: auto-desliga após esse tempo
	function obterTempoSelecionadoSegundos() {
		// Se o select não existir, tratamos como null.
		if (!selTempo) return null;

		// value="" representa o caso null (sem auto-desligar).
		const v = String(selTempo.value || '').trim();
		if (!v) return null;

		const n = parseInt(v, 10);
		if (!Number.isFinite(n) || n <= 0) return null;
		return n;
	}

	// Para e limpa qualquer temporizador/contador do STT.
	function limparTemporizadoresSTT() {
		if (sttAutoStopTimeoutId) {
			clearTimeout(sttAutoStopTimeoutId);
			sttAutoStopTimeoutId = null;
		}
		if (sttCountdownIntervalId) {
			clearInterval(sttCountdownIntervalId);
			sttCountdownIntervalId = null;
		}
		sttAutoStopFimMs = 0;
		setContador('');
	}

	// Inicia o contador regressivo e agenda o auto-desligamento.
	function iniciarTemporizadorSTT(segundos) {
		limparTemporizadoresSTT();
		const s = Number(segundos);
		if (!Number.isFinite(s) || s <= 0) {
			setContador('');
			return;
		}

		sttAutoStopFimMs = Date.now() + s * 1000;

		// Atualiza o contador imediatamente.
		atualizarContadorRegressivo();

		// Atualiza a cada 200ms (mostrando segundos inteiros, sem ficar atrasando).
		sttCountdownIntervalId = setInterval(atualizarContadorRegressivo, 200);

		// Agenda o auto-desligamento.
		sttAutoStopTimeoutId = setTimeout(() => {
			// Só desliga se ainda estiver ativo.
			if (sttAtivo) window.desativarSTT();
		}, s * 1000);
	}

	// Atualiza o texto do contador regressivo com base no tempo final.
	function atualizarContadorRegressivo() {
		if (!sttAutoStopFimMs) {
			setContador('');
			return;
		}

		const msRestante = sttAutoStopFimMs - Date.now();
		const segRestante = Math.max(0, Math.ceil(msRestante / 1000));
		setContador(`Contador: ${segRestante}s`);

		// Quando chegar em 0, paramos o intervalo para não ficar rodando.
		if (segRestante <= 0 && sttCountdownIntervalId) {
			clearInterval(sttCountdownIntervalId);
			sttCountdownIntervalId = null;
		}
	}

	// Normaliza um texto para facilitar a comparação.
	// Ex.: "Olá,   mundo!" -> "ola mundo"
	function normalizarTexto(texto) {
		// Garante que é string.
		let s = String(texto || '');

		// Tira espaços no começo/fim.
		s = s.trim();

		// Coloca tudo em minúsculas.
		s = s.toLowerCase();

		// Remove acentos (funciona para português e francês).
		// "ação" -> "acao" | "français" -> "francais"
		s = s.normalize('NFD').replace(/[\u0300-\u036f]/g, '');

		// Remove pontuação e símbolos (fica só letras/números/espaços).
		s = s.replace(/[^a-z0-9\s]/g, ' ');

		// Troca vários espaços por um só.
		s = s.replace(/\s+/g, ' ').trim();

		return s;
	}

	// Converte números "por extenso" para "numeral" (ou mantém se já for numeral).
	// Exemplo francês: "un" -> "1" | "deux" -> "2"
	// Exemplo português: "um" -> "1" | "dois" -> "2"
	// Isso serve para o caso: você falar "un" e o texto esperado estar "1" (ou vice-versa).
	function canonizarNumeros(textoNormalizado, lang) {
		// O texto aqui já deve estar normalizado (minúsculo, sem acentos, sem pontuação).
		const s = String(textoNormalizado || '').trim();
		if (!s) return '';

		// Decide qual dicionário usar pelo idioma do combobox.
		const langStr = String(lang || '').toLowerCase();
		const ehFrances = langStr.startsWith('fr');

		// Dicionário simples (0 a 10) — você pode ampliar depois.
		const mapaFR = {
			zero: '0',
			un: '1',
			une: '1',
			deux: '2',
			trois: '3',
			quatre: '4',
			cinq: '5',
			six: '6',
			sept: '7',
			huit: '8',
			neuf: '9',
			dix: '10'
		};

		const mapaPT = {
			zero: '0',
			um: '1',
			uma: '1',
			dois: '2',
			duas: '2',
			tres: '3',
			quatro: '4',
			cinco: '5',
			seis: '6',
			sete: '7',
			oito: '8',
			nove: '9',
			dez: '10'
		};

		const mapa = ehFrances ? mapaFR : mapaPT;

		// Quebra em palavras e troca as que existem no dicionário.
		// Ex.: "bonjour un" -> ["bonjour","un"] -> ["bonjour","1"]
		const tokens = s.split(' ');
		const tokensCanon = tokens.map((tok) => (mapa[tok] ? mapa[tok] : tok));
		return tokensCanon.join(' ');
	}

	// Calcula a distância de Levenshtein entre duas strings.
	// Ideia simples: quantas edições (inserir/apagar/trocar) para virar uma na outra.
	function distanciaLevenshtein(a, b) {
		const s = String(a);
		const t = String(b);

		// Casos rápidos.
		if (s === t) return 0;
		if (!s) return t.length;
		if (!t) return s.length;

		// Usamos programação dinâmica com duas linhas (economiza memória).
		const cols = t.length + 1;
		let prev = new Array(cols);
		let curr = new Array(cols);

		// Linha inicial: transformar string vazia em t.
		for (let j = 0; j < cols; j++) prev[j] = j;

		for (let i = 1; i <= s.length; i++) {
			// Coluna inicial: transformar s[0..i] em vazio.
			curr[0] = i;
			const sChar = s.charCodeAt(i - 1);

			for (let j = 1; j <= t.length; j++) {
				const custoTroca = sChar === t.charCodeAt(j - 1) ? 0 : 1;

				// min(inserir, apagar, trocar)
				curr[j] = Math.min(
					curr[j - 1] + 1,
					prev[j] + 1,
					prev[j - 1] + custoTroca
				);
			}

			// Avança: a linha atual vira a anterior.
			[prev, curr] = [curr, prev];
		}

		// No fim, a resposta está em prev[última coluna].
		return prev[t.length];
	}

	// Compara a última fala reconhecida com o texto digitado.
	function compararUltimaFala() {
		// Pega o que o usuário digitou no input.
		const esperadoOriginal = txtEsperado ? txtEsperado.value : '';
		const ditoOriginal = ultimaTranscricaoFinal;

		// Se ainda não existe fala reconhecida, avisa.
		if (!ditoOriginal.trim()) {
			setStatus('Ainda não tenho uma transcrição final. Fale algo primeiro.');
			return;
		}

		// Se o usuário não digitou nada, avisa.
		if (!esperadoOriginal.trim()) {
			setStatus('Digite no campo "Texto esperado" e clique em Comparar.');
			return;
		}

		// Normaliza os dois textos para comparar de forma mais justa.
		const esperadoNorm = normalizarTexto(esperadoOriginal);
		const ditoNorm = normalizarTexto(ditoOriginal);

		// Aplica a regra: número por extenso e numeral devem ser equivalentes.
		// Ex.: esperado "1" e dito "un" => ambos viram "1".
		const langAtual = obterIdiomaSelecionado();
		const esperado = canonizarNumeros(esperadoNorm, langAtual);
		const dito = canonizarNumeros(ditoNorm, langAtual);

		// Se depois de normalizar ficou vazio, avisa.
		if (!esperado) {
			setStatus('O texto esperado ficou vazio após normalização. Digite letras/números.');
			return;
		}

		// Calcula uma “nota” de similaridade de 0 a 1.
		// 1 = igual, 0 = totalmente diferente.
		const dist = distanciaLevenshtein(esperado, dito);
		const maxLen = Math.max(esperado.length, dito.length) || 1;
		const similaridade = 1 - dist / maxLen;

		// Classificação simples:
		// - certo: textos iguais após normalizar
		// - meio certo: parecido (>= 0.75)
		// - errado: abaixo disso
		let resultado;
		if (esperado === dito) resultado = 'CERTO';
		else if (similaridade >= 0.75) resultado = 'MEIO CERTO';
		else resultado = 'ERRADO';

		// Mostra um resumo no status e também registra no painel de saída.
		const pct = Math.round(similaridade * 100);
		setStatus(`Comparação: ${resultado} (${pct}%)`);
		appendSaida(`Comparar -> esperado: "${esperadoOriginal}" | dito: "${ditoOriginal}" => ${resultado} (${pct}%)`);
	}

	// Função auxiliar: escreve texto no status (se o elemento existir).
	function setStatus(texto) {
		// Se statusEl existir, atualiza o texto visível.
		if (statusEl) statusEl.textContent = texto;
	}

	// Função auxiliar: adiciona uma “linha” de transcrição no painel de saída.
	function appendSaida(texto) {
        
		// Se não existe o elemento de saída, não faz nada.
		if (!saidaEl) return;

		// Cria um <div> para ser uma linha.
		const linha = document.createElement('div');

		// Coloca o texto dentro do <div>.
		linha.textContent = texto;

		// Anexa a linha dentro do painel #saida.
		saidaEl.appendChild(linha);
	}

	// Limpa todo o conteúdo do painel de saída.
	function limparSaida() {
		if (!saidaEl) return;
		saidaEl.innerHTML = '';
		if (txtTraduzido) txtTraduzido.value = '';
		setStatus('Saída limpa.');
	}

	// Mostra/esconde o campo de tradução.
	function setCampoTraduzidoVisivel(visivel) {
		if (!wrapTraduzido) return;
		wrapTraduzido.style.display = visivel ? '' : 'none';
	}

	// Cache local de vozes do navegador (TTS).
	let vozesCache = [];

	// Atualiza a lista de vozes disponíveis.
	function atualizarVozesCache() {
		// `speechSynthesis.getVoices()` retorna um array de vozes instaladas/disponíveis.
		// Em alguns navegadores ele pode lançar exceção (por isso o try/catch).
		try {
			vozesCache = window.speechSynthesis ? window.speechSynthesis.getVoices() : [];
		} catch (_) {
			// Se falhar por qualquer motivo, tratamos como “sem vozes”.
			vozesCache = [];
		}
	}

	// Aguarda as vozes estarem disponíveis (alguns navegadores carregam depois).
	function aguardarVozesCarregarem(timeoutMs = 1200) {
		// Motivo desta função:
		// - Em alguns navegadores, `getVoices()` vem vazio no carregamento da página.
		// - Depois de alguns ms, o navegador dispara `voiceschanged` e aí as vozes aparecem.
		// - Sem esperar, a gente escolhe "nenhuma voz" e o TTS pode cair no inglês.
		return new Promise((resolve) => {
			if (!window.speechSynthesis) return resolve(false);
			atualizarVozesCache();
			if (vozesCache && vozesCache.length > 0) return resolve(true);

			// Proteção para garantir que resolvemos a Promise uma única vez.
			let resolvido = false;
			const finalizar = (ok) => {
				if (resolvido) return;
				resolvido = true;
				try {
					// Remove o listener temporário para evitar vazamento de eventos.
					window.speechSynthesis.removeEventListener('voiceschanged', onChange);
				} catch (_) {
					// ignora
				}
				resolve(ok);
			};

			// Handler chamado quando o navegador sinaliza que a lista de vozes mudou.
			const onChange = () => {
				atualizarVozesCache();
				if (vozesCache && vozesCache.length > 0) finalizar(true);
			};

			try {
				window.speechSynthesis.addEventListener('voiceschanged', onChange);
			} catch (_) {
				// Se não der para escutar evento, só usa timeout.
			}

			// Timeout: mesmo sem evento, tentamos de novo após alguns ms.
			setTimeout(() => {
				atualizarVozesCache();
				finalizar(!!(vozesCache && vozesCache.length > 0));
			}, timeoutMs);
		});
	}

	// Tenta escolher a melhor "voz" para o idioma selecionado.
	function escolherVozPorIdioma(langDesejado) {
		// Entrada típica: "fr-FR" ou "pt-BR" (vem do combobox).
		// Estratégia:
		// 1) Pontuar cada voz candidata (idioma exato ganha mais pontos)
		// 2) Penalizar vozes "Multilingual" (muitas vezes pronunciam números em inglês)
		// 3) Preferir vozes do país certo quando o nome indica (France/Brazil)
		const lang = String(langDesejado || '').toLowerCase();
		if (!lang) return null;

		// Garante que temos vozes no cache (alguns navegadores carregam isso depois).
		if (!vozesCache || vozesCache.length === 0) atualizarVozesCache();
		if (!vozesCache || vozesCache.length === 0) return null;

		// Idioma-base: "fr" de "fr-FR", "pt" de "pt-BR".
		const base = lang.split('-')[0];

		// Função de pontuação: quanto maior o score, melhor.
		// A ideia é escolher consistentemente uma voz "boa" para o idioma.
		function pontuarVoz(v) {
			const vLang = String(v.lang || '').toLowerCase();
			const vName = String(v.name || '').toLowerCase();
			let score = 0;

			// Pontos por compatibilidade de idioma.
			// - Igual exato (fr-FR) é o melhor caso.
			// - fr-* é aceitável, mas abaixo do exato.
			// - começar com fr (qualquer coisa) é o fallback.
			if (vLang === lang) score += 100;
			else if (vLang.startsWith(base + '-')) score += 60;
			else if (vLang.startsWith(base)) score += 50;

			// Penaliza vozes multilíngues: elas podem “puxar” pronúncia para inglês.
			if (vName.includes('multilingual') || vName.includes('multi-lingual')) score -= 40;

			// Ajuda a preferir o país certo quando existe (ex.: French (France) para fr-FR).
			if (lang === 'fr-fr' && (vName.includes('french (france)') || vName.includes('france'))) score += 15;
			if (lang === 'pt-br' && (vName.includes('portuguese (brazil)') || vName.includes('brazil') || vName.includes('brasil'))) score += 15;

			// Pequena preferência para a voz default do sistema.
			if (v.default) score += 5;
			return score;
		}

		// Filtra apenas vozes que parecem compatíveis com o idioma desejado.
		// Ex.: se o usuário escolheu fr-FR, aceitamos fr-FR, fr-CA, fr-BE, etc.
		const candidatas = vozesCache.filter((v) => {
			const vLang = String(v.lang || '').toLowerCase();
			return vLang === lang || vLang.startsWith(base);
		});
		if (candidatas.length > 0) {
			// Varre todas as candidatas e pega a de maior score.
			let melhor = candidatas[0];
			let melhorScore = pontuarVoz(melhor);
			for (let i = 1; i < candidatas.length; i++) {
				const c = candidatas[i];
				const s = pontuarVoz(c);
				if (s > melhorScore) {
					melhor = c;
					melhorScore = s;
				}
			}
			return melhor;
		}

		// Fallback por nome (algumas vozes têm lang estranho, mas o nome denuncia).
		// Ex.: a voz pode vir com `lang` incompleto, mas o `name` contém "French".
		const nomeAlvo = base === 'fr' ? 'french|francais|fran' : base === 'pt' ? 'portuguese|portugues|portugu' : '';
		if (nomeAlvo) {
			const re = new RegExp(nomeAlvo, 'i');
			const porNome = vozesCache.filter((v) => re.test(String(v.name || '')));
			if (porNome.length > 0) {
				// Mesmo no fallback por nome, escolhemos a melhor pelo score.
				let melhor = porNome[0];
				let melhorScore = pontuarVoz(melhor);
				for (let i = 1; i < porNome.length; i++) {
					const c = porNome[i];
					const s = pontuarVoz(c);
					if (s > melhorScore) {
						melhor = c;
						melhorScore = s;
					}
				}
				return melhor;
			}
		}

		return null;
	}

	// Mostra no painel as vozes disponíveis (útil para diagnosticar idioma/voz no Windows).
	function diagnosticarVozesTTS(langSelecionado) {
		if (!window.speechSynthesis) {
			appendSaida('TTS: speechSynthesis indisponível.');
			return;
		}

		atualizarVozesCache();
		const total = Array.isArray(vozesCache) ? vozesCache.length : 0;
		const lang = String(langSelecionado || '').toLowerCase();
		const base = lang.split('-')[0];

		appendSaida(`TTS: idioma selecionado = ${langSelecionado} | vozes carregadas = ${total}`);
		if (!total) {
			appendSaida('TTS: nenhuma voz carregada ainda (tente recarregar a página).');
			return;
		}

		// Lista vozes do idioma-base (ex.: fr-*) para confirmar se existe voz instalada.
		const vocesBase = vozesCache
			.filter((v) => String(v.lang || '').toLowerCase().startsWith(base))
			.slice(0, 12);

		if (vocesBase.length === 0) {
			appendSaida(`TTS: não encontrei vozes com lang começando por "${base}" (ex.: ${base}-XX).`);
			appendSaida('TTS: nesse caso o navegador costuma cair no inglês/default.');
			return;
		}

		appendSaida(`TTS: vozes ${base}* encontradas (até 12):`);
		for (const v of vocesBase) {
			appendSaida(`- ${String(v.name || '(sem nome)')} | lang=${String(v.lang || '')}${v.default ? ' | default' : ''}`);
		}
	}

	// Faz o navegador falar um texto (TTS) no idioma selecionado.
	async function falarTexto(texto) {
		const textoLimpo = String(texto || '').trim();
		if (!textoLimpo) {
			setStatus('Nada para falar ainda. Fale algo e espere sair uma transcrição final.');
			return;
		}

		// Verifica suporte ao TTS.
		if (!window.speechSynthesis || typeof window.SpeechSynthesisUtterance !== 'function') {
			appendSaida('ERRO: TTS indisponível neste navegador.');
			return;
		}

		// Para qualquer fala anterior para não “misturar”.
		try {
			window.speechSynthesis.cancel();
		} catch (_) {
			// Se falhar, seguimos mesmo assim.
		}

		// Em alguns navegadores, um pequeno "tick" ajuda a efetivar o cancel antes do speak.
		await new Promise((r) => setTimeout(r, 0));

		// Cria a fala (utterance) com o texto.
		const fala = new SpeechSynthesisUtterance(textoLimpo);

		// Aplica a velocidade do áudio conforme o combobox.
		fala.rate = obterTtsRateSelecionado();

		// Usa o mesmo idioma escolhido no combobox (pt-BR ou fr-FR).
		const langSelecionado = obterIdiomaSelecionado();
		fala.lang = langSelecionado;

		// Também tenta escolher uma VOZ que combine com o idioma.
		// OBS: só setar fala.lang nem sempre muda a voz automaticamente.
		// Primeiro, tenta garantir que a lista de vozes já carregou.
		await aguardarVozesCarregarem();
		// Diagnóstico: mostra quais vozes existem para o idioma-base.
		diagnosticarVozesTTS(langSelecionado);
		const voz = escolherVozPorIdioma(langSelecionado);
		if (voz) {
			fala.voice = voz;
			appendSaida(`TTS: usando voz "${String(voz.name || '')}" | lang=${String(voz.lang || '')}`);
		} else {
			// Se não existe voz no idioma, NÃO falamos, para não cair no inglês.
			appendSaida(`ERRO: não existe voz TTS instalada para ${langSelecionado}.`);
			setStatus(`Instale uma voz de ${langSelecionado} no sistema/navegador e tente novamente.`);
			return;
		}

		fala.onstart = () => {
			setStatus(`Falando (${langSelecionado})...`);
		};
		fala.onend = () => {
			setStatus('Fala concluída.');
		};
		fala.onerror = (ev) => {
			appendSaida(`ERRO TTS: ${String((ev && ev.error) || 'desconhecido')}`);
			setStatus('Erro ao falar.');
		};

		// Dispara o TTS.
		try {
			window.speechSynthesis.speak(fala);
		} catch (e) {
			appendSaida(`ERRO: falha ao chamar speechSynthesis.speak(): ${String(e && e.message ? e.message : e)}`);
		}
	}

	// Atualiza botão e status conforme o estado atual e suporte do navegador.
	function atualizarTela() {
		// Se o botão existe, muda o texto dependendo de sttAtivo.
		if (btnSTT) btnSTT.textContent = sttAtivo ? 'Desativar STT' : 'Ativar STT';

		// Se o navegador não suporta SpeechRecognition, avisa e desabilita o botão.
		if (!SpeechRecognitionImpl) {
			// Mostra uma mensagem de erro amigável.
			setStatus('STT indisponível: este navegador não suporta SpeechRecognition.');

			// Se o botão existe, desabilita para evitar clique sem efeito.
			if (btnSTT) btnSTT.disabled = true;

			// Sai da função; nada mais para fazer.
			return;
		}

		// Se há suporte, mostra o status de ativo/desativado.
		setStatus(sttAtivo ? 'STT ativo (escutando...)' : 'STT desativado');
	}

	// Cria a instância de SpeechRecognition (apenas uma vez), se necessário.
	function criarRecognitionSePreciso() {
		// Se já criamos a instância antes, não cria de novo.
		if (recognition) return;

		// Se não existe implementação no navegador, não cria.
		if (!SpeechRecognitionImpl) return;

		// Cria a instância do reconhecimento de fala.
		recognition = new SpeechRecognitionImpl();

		// Faz o reconhecimento ficar contínuo (escuta até você mandar parar).
		recognition.continuous = true;

		// Permite resultados parciais (interim) enquanto você fala.
		recognition.interimResults = true;

		// Define um idioma inicial (pode ser sobrescrito antes do start).
		// OBS: o idioma “que vale” é o que estiver em recognition.lang quando chamar recognition.start().
		recognition.lang = 'pt-BR';

		// Evento disparado quando o reconhecimento começa de fato.
		recognition.onstart = () => {
			// Marca como ativo.
			sttAtivo = true;

			// Se o usuário pediu um tempo (em segundos), inicia o temporizador SOMENTE agora.
			// Isso garante que o tempo conte a partir de quando o STT realmente começou.
			if (sttTempoSolicitadoSegundos != null) {
				iniciarTemporizadorSTT(sttTempoSolicitadoSegundos);
			} else {
				limparTemporizadoresSTT();
			}

			// Atualiza a UI.
			atualizarTela();
		};

		// Evento disparado quando o reconhecimento encerra.
		recognition.onend = () => {
			// Alguns navegadores param automaticamente após silêncio.
			// Aqui apenas refletimos o estado final como desligado.
			sttAtivo = false;

			// Ao terminar, limpa o contador/timers.
			limparTemporizadoresSTT();

			// Atualiza a UI.
			atualizarTela();
		};

		// Evento de erro (permite mostrar a causa, como "not-allowed").
		recognition.onerror = (event) => {
			// Extrai o tipo do erro se existir.
			const erro = event && event.error ? String(event.error) : 'erro-desconhecido';
			const mensagem = event && event.message ? String(event.message) : '';
			const langAtual = (recognition && recognition.lang) ? String(recognition.lang) : obterIdiomaSelecionado();

			// Mostra o erro no painel de saída com mais contexto (didático).
			appendSaida(
				`ERRO SpeechRecognition: ${erro}` +
				(mensagem ? ` | msg: ${mensagem}` : '') +
				(langAtual ? ` | lang: ${langAtual}` : '')
			);

			// Mensagens de ajuda por tipo de erro.
			// Lista de erros mais comuns (padrão Web Speech API):
			// - not-allowed / service-not-allowed: permissão negada
			// - no-speech: não detectou fala
			// - audio-capture: sem microfone
			// - network: falha na rede/serviço
			// - language-not-supported: idioma não suportado pelo navegador
			let dica = '';
			switch (erro) {
				case 'not-allowed':
				case 'service-not-allowed':
					dica = 'Permissão do microfone negada. Libere o microfone no cadeado do navegador e recarregue.';
					break;
				case 'no-speech':
					dica = 'Não detectei fala. Fale mais perto do microfone e aguarde 1–2s.';
					break;
				case 'audio-capture':
					dica = 'Não achei microfone. Verifique se o microfone está conectado e selecionado no sistema.';
					break;
				case 'network':
					dica = 'Falha de rede/serviço do STT. Teste em Chrome/Edge e em HTTPS/localhost.';
					break;
				case 'language-not-supported':
					dica = `Este navegador não suporta STT para ${langAtual}. Troque o idioma do STT ou use Chrome/Edge com suporte.`;
					break;
				case 'aborted':
					dica = 'Reconhecimento interrompido (abort/stop).';
					break;
				default:
					dica = 'Veja a saída para detalhes e tente novamente.';
					break;
			}
			setStatus(`Erro no STT: ${erro}. ${dica}`);

			// Em caso de erro, é mais seguro considerar como desligado.
			sttAtivo = false;

			// Em caso de erro, também limpamos o contador/timers.
			limparTemporizadoresSTT();

			// Atualiza a UI.
			atualizarTela();
		};

		// Evento com resultados (parciais e finais) do reconhecimento.
		recognition.onresult = (event) => {
			// Texto final acumulado (quando o navegador marca isFinal).
			let finalText = '';

			// Texto parcial (interim) acumulado enquanto você fala.
			let interimText = '';

			// Percorre os resultados novos, começando de resultIndex.
			for (let i = event.resultIndex; i < event.results.length; i++) {
				// Resultado atual.
				const result = event.results[i];

				// Transcrição com fallback para string vazia.
				const transcript = result[0] && result[0].transcript ? result[0].transcript : '';

				// Se é final, vai para finalText; se não, vai para interimText.
				if (result.isFinal) finalText += transcript;
				else interimText += transcript;
			}

			// Se existe texto parcial, mostra no status (útil para “ver” enquanto fala).
			if (interimText.trim()) {
				setStatus(`Ouvindo: ${interimText.trim()}`);
			}

			// Se existe texto final, adiciona uma linha e volta o status para “ativo”.
			if (finalText.trim()) {
				// Guarda a última transcrição final para o botão "Falar".
				ultimaTranscricaoFinal = finalText.trim();

				// Adiciona a transcrição final ao painel.
				appendSaida(finalText.trim());

				// Volta a mensagem de status para o padrão de ativo.
				setStatus('STT ativo (escutando...)');
			}
		};
	}

	// Lê o idioma escolhido no combobox.
	function obterIdiomaSelecionado() {
		// Se o select existir e tiver valor, usa ele; senão, usa pt-BR como padrão.
		// IMPORTANTE: o value do select é um código BCP-47 (ex.: fr-FR, en-US, pt-PT...).
		return selLang && selLang.value ? String(selLang.value) : 'pt-BR';
	}

	// Lê o idioma NATIVO (origem) da pessoa.
	// Esse select existe para o caso em que o usuário pensa/escreve no idioma nativo,
	// mas quer ouvir a tradução no idioma destino.
	function obterIdiomaNativoSelecionado() {
		return selLangNativo && selLangNativo.value ? String(selLangNativo.value) : 'pt-BR';
	}

	// Lê a velocidade do TTS no combobox (0.50, 0.75, 1.00, 1.25...).
	function obterTtsRateSelecionado() {
		if (!selTtsRate) return 1;
		const v = String(selTtsRate.value || '').trim();
		const n = parseFloat(v);
		if (!Number.isFinite(n) || n <= 0) return 1;
		// Range aceito costuma ser 0.1..10 (varia por navegador, mas é seguro limitar).
		return Math.max(0.1, Math.min(10, n));
	}

	// Converte BCP-47 para um código base (muito usado em APIs de tradução).
	// Ex.: "fr-FR" -> "fr" | "pt-BR" -> "pt"
	function bcp47ParaBase(bcp47) {
		const s = String(bcp47 || '').trim();
		if (!s) return '';
		return s.split('-')[0].toLowerCase();
	}

	// Descobre qual URL usar para chamar o servidor de tradução.
	// Caso você abra este HTML por outro servidor (ex.: Live Server em outra porta),
	// um fetch relativo para "/api/translate" vai bater no servidor errado e pode
	// retornar HTTP 405 (Method Not Allowed).
	//
	// Estratégia:
	// - Se esta página já está no servidor independente (porta 3100), usamos URL relativa.
	// - Se esta página estiver em outro host/porta (ex.: XAMPP/Apache em :80),
	//   chamamos o MESMO host na porta 3100 (ex.: http://meusite:3100).
	// - Se estiver em file:// (sem host), cai em http://localhost:3100.
	//
	// Override opcional (quando você quer controlar manualmente sem depender de heurística):
	// - Defina no HTML: window.TRANSLATE_API_BASE = "http://SEU_HOST:3100";
	// - Ou via meta tag: <meta name="translate-api-base" content="http://SEU_HOST:3100">
	// - Para forçar relativo (mesma origem), use: "relative".
	function obterBaseApiTraducao() {
		// 0) Override manual (se existir)
		try {
			const forcedGlobal = typeof window.TRANSLATE_API_BASE === 'string' ? window.TRANSLATE_API_BASE : '';
			const forcedMeta = document.querySelector('meta[name="translate-api-base"]')?.getAttribute('content') || '';
			const forced = String(forcedGlobal || forcedMeta || '').trim();
			if (forced) return forced.toLowerCase() === 'relative' ? '' : forced;
		} catch (_) {
			// ignora e segue a heurística
		}

		// Se a página está em HTTPS (ex.: domínio público), não podemos chamar HTTP
		// (mixed content). Nesse caso, a forma certa é o servidor web (Apache/IIS)
		// fazer proxy para o Node em 3100, e o navegador chamar a URL relativa.
		const proto = String(window.location.protocol || '').toLowerCase();
		if (proto === 'https:') return '';

		const host = String(window.location.hostname || '').trim();
		const port = String(window.location.port || '').trim();
		// Se já estamos no servidor independente (porta 3100), usamos URL relativa.
		if (port === '3100') return '';
		// Se estiver em file:// (ou algo sem hostname), não dá para inferir o host.
		if (!host) return 'http://localhost:3100';
		// Em produção (ex.: XAMPP/Apache), use o mesmo host na porta 3100.
		// Observação: se sua página estiver em HTTPS, o navegador pode bloquear chamada HTTP.
		return `http://${host}:3100`;
	}

	// Chama o servidor local para traduzir um texto.
	// Observação: o navegador NÃO traduz sozinho; precisamos de um serviço.
	// Por isso existe o endpoint `/api/translate` no server.js.
	async function traduzirNoServidor(texto, origemBcp47, destinoBcp47) {
		const text = String(texto || '').trim();
		if (!text) throw new Error('Texto vazio.');

		const source = bcp47ParaBase(origemBcp47);
		const target = bcp47ParaBase(destinoBcp47);
		if (!source || !target) throw new Error('Idioma inválido (origem/destino).');
		// Caso comum: pt-BR e pt-PT viram ambos "pt".
		// Não existe "tradução" entre variantes aqui; evitamos erro no backend.
		if (source === target) return text;

		// Tentamos duas rotas possíveis:
		// 1) relativa à pasta atual (ex.: /aprender/api/translate) — ideal para Apache/IIS com proxy
		// 2) raiz do host (ex.: /api/translate) — útil quando o Node serve na raiz
		// 3) host explícito (ex.: http://host:3100/api/translate) — útil em HTTP
		const base = obterBaseApiTraducao();
		const urls = [];
		urls.push('api/translate');
		urls.push('/api/translate');
		if (base) urls.push(`${base}/api/translate`);

		/** @type {Response|null} */
		let resp = null;
		/** @type {string} */
		let lastFetchError = '';
		let urlUsada = '';

		for (const url of urls) {
			urlUsada = url;
			try {
				resp = await fetch(url, {
					method: 'POST',
					mode: 'cors',
					headers: { 'Content-Type': 'application/json' },
					body: JSON.stringify({ text, source, target })
				});
				// Se deu 405, pode ser o “servidor errado”; tentamos o próximo.
				if (resp.status === 405) continue;
				break;
			} catch (e) {
				lastFetchError = e && e.message ? String(e.message) : String(e);
				resp = null;
				continue;
			}
		}

		if (!resp) {
			throw new Error(
				`Não foi possível conectar ao servidor de tradução. ` +
				(
					String(window.location.protocol || '').toLowerCase() === 'https:'
						? `Dica: como sua página está em HTTPS, configure proxy no Apache/IIS para enviar /aprender/api/translate -> http://127.0.0.1:3100/api/translate. `
						: `Suba o servidor (Node 3100) e abra a página por ele (ex.: http://SEU_HOST:3100/microfonestt.html). `
				) +
				(lastFetchError ? `Detalhe: ${lastFetchError}` : '')
			);
		}

		if (!resp.ok) {
			// Tenta obter o máximo de detalhe possível para você entender o motivo.
			let msg = `Falha na tradução (HTTP ${resp.status}).`;
			if (resp.status === 405) {
				msg +=
					' Dica: este erro normalmente acontece quando a página NÃO está sendo servida pelo server.js (porta 3000).';
			}
			msg += ` URL: ${urlUsada}`;
			try {
				const j = await resp.json();
				if (j && j.message) msg = String(j.message);
				if (j && j.detail) msg += ` Detalhe: ${String(j.detail)}`;
			} catch (_) {
				// Se não for JSON, tenta ler como texto.
				try {
					const t = await resp.text();
					if (t && t.trim()) msg += ` Detalhe: ${t.trim()}`;
				} catch (_) {
					// ignora
				}
			}
			throw new Error(msg);
		}

		const data = await resp.json();
		// Extra (didático): alguns servidores retornam tempos (timingsMs) para explicar lentidão.
		// Não é obrigatório para funcionar, mas ajuda no aprendizado.
		if (data && data.timingsMs) {
			try {
				appendSaida(`(timings) ${JSON.stringify(data.timingsMs)}`);
			} catch (_) {
				// ignora
			}
		}
		const translatedText = data && data.translatedText ? String(data.translatedText) : '';
		if (!translatedText.trim()) throw new Error('Tradução vazia.');
		return translatedText;
	}

	// Clique do botão Traduzir.
	// Requisito: traduz do idioma nativo (origem) e fala no idioma destino.
	async function aoClicarTraduzir() {
		const textoDigitado = txtEsperado ? String(txtEsperado.value || '') : '';
		const texto = textoDigitado.trim();
		if (!texto) {
			setStatus('Digite um texto no campo para traduzir.');
			return;
		}

		// Requisito: o edit de “Texto traduzido” deve aparecer quando mandar traduzir.
		setCampoTraduzidoVisivel(true);
		if (txtTraduzido) txtTraduzido.value = '';

		const origem = obterIdiomaNativoSelecionado();
		const destino = obterIdiomaSelecionado();

		appendSaida(`Traduzir -> origem: ${origem} | destino: ${destino} | texto: "${texto}"`);
		setStatus(`Traduzindo (${origem} -> ${destino})... (se for 1ª vez desse idioma, pode baixar modelo e demorar)`);

		try {
			const traduzido = await traduzirNoServidor(texto, origem, destino);
			appendSaida(`Tradução -> "${traduzido}"`);
			if (txtTraduzido) txtTraduzido.value = traduzido;

			// Fala no idioma DESTINO (o falarTexto escolhe voz conforme obterIdiomaSelecionado()).
			setStatus('Tradução pronta. Falando...');
			await falarTexto(traduzido);
		} catch (e) {
			appendSaida(`ERRO ao traduzir: ${e && e.message ? e.message : e}`);
			// Mantém o campo visível (foi pedido para aparecer ao traduzir), mas sem texto.
			if (txtTraduzido) txtTraduzido.value = '';
			setStatus('Erro ao traduzir. Veja a saída.');
		}
	}

	// Se o usuário mudar o idioma enquanto o STT está ativo, desativa o STT.
	function aoMudarIdiomaDesativarSTT() {
		// Se estiver escutando, desliga para garantir que o novo idioma só valha no próximo start().
		if (sttAtivo) {
			window.desativarSTT();
			appendSaida(`(Idioma alterado para ${obterIdiomaSelecionado()}) STT foi desativado.`);
			setStatus('Idioma alterado: STT desativado. Ative novamente.');
			return;
		}

		// Se não estiver ativo e já existir recognition, podemos só atualizar o lang para o próximo uso.
		if (recognition) {
			recognition.lang = obterIdiomaSelecionado();
		}
	}

	// ===== Funções pedidas: ativação e desativação para MODO STT =====

	// Expõe uma função global para ligar o STT.
	window.ativarSTT = function ativarSTT() {
		// Se não há suporte, só atualiza a tela e retorna false.
		if (!SpeechRecognitionImpl) {
			atualizarTela();
			return false;
		}

		// Garante que a instância exista.
		criarRecognitionSePreciso();

		// Se mesmo assim não temos recognition, falha.
		if (!recognition) return false;

		// Define o idioma escolhido ANTES de iniciar o reconhecimento.
		recognition.lang = obterIdiomaSelecionado();

		// Lê o tempo escolhido no combobox e guarda para o onstart.
		// - null: não auto-desliga
		// - número: auto-desliga após esse tempo
		sttTempoSolicitadoSegundos = obterTempoSelecionadoSegundos();
		if (sttTempoSolicitadoSegundos == null) {
			setContador('');
		} else {
			setContador(`Contador: ${sttTempoSolicitadoSegundos}s`);
		}

		// Tenta dar start no reconhecimento.
		try {
			// Inicia o reconhecimento (o navegador pode pedir permissão aqui).
			recognition.start();

			// onstart vai marcar sttAtivo e atualizar a UI.
			return true;
		} catch (e) {
			// start() pode lançar exceção (por exemplo, se já estiver ativo).
			appendSaida(`ERRO ao ativar STT: ${e && e.message ? e.message : e}`);
			// Se não conseguiu iniciar, não faz sentido manter timers.
			limparTemporizadoresSTT();
			return false;
		}
	};

	// Expõe uma função global para desligar o STT.
	window.desativarSTT = function desativarSTT() {
		// Qualquer desligamento (manual/auto) cancela o temporizador.
		limparTemporizadoresSTT();
		sttTempoSolicitadoSegundos = null;

		// Se nem existe recognition, só garante estado/ UI e retorna true.
		if (!recognition) {
			sttAtivo = false;
			atualizarTela();
			return true;
		}

		// Tenta parar “normalmente”.
		try {
			// stop() tenta finalizar e entregar resultados pendentes.
			recognition.stop();
			return true;
		} catch (e) {
			// Se stop() falhar, tenta abort() (encerra imediatamente).
			try {
				recognition.abort();
				return true;
			} catch (e2) {
				// Se até abort falhar, registra erro.
				appendSaida(`ERRO ao desativar STT: ${e2 && e2.message ? e2.message : e2}`);
				return false;
			}
		}
	};

	// Função local: alterna entre ativar e desativar.
	function toggleSTT() {
		// Se já está ativo, desativa; se não, ativa.
		if (sttAtivo) window.desativarSTT();
		else window.ativarSTT();
	}

	// Se o botão existir no HTML, liga o click ao toggle.
	if (btnSTT) btnSTT.addEventListener('click', toggleSTT);

	// Se o combobox existir, quando mudar o idioma aplica a regra de desativar.
	if (selLang) selLang.addEventListener('change', aoMudarIdiomaDesativarSTT);

	// Botão Traduzir: traduz e fala.
	if (btnTraduzir) btnTraduzir.addEventListener('click', () => {
		// Dispara sem bloquear a UI.
		aoClicarTraduzir();
	});

	// Clique do botão "Falar": fala a última transcrição final.
	if (btnFalar) btnFalar.addEventListener('click', () => {
		// Não precisamos await aqui; é só disparar.
		falarTexto(ultimaTranscricaoFinal);
	});

	// Alguns navegadores carregam as vozes TTS com atraso.
	// Este evento avisa quando a lista de vozes muda (ex.: quando termina de carregar).
	if (window.speechSynthesis) {
		// Primeira leitura imediata (pode vir vazia em alguns navegadores).
		atualizarVozesCache();
		// Listener permanente: quando o navegador terminar de carregar as vozes,
		// ele dispara `voiceschanged` e a gente atualiza o cache.
		try {
			window.speechSynthesis.addEventListener('voiceschanged', atualizarVozesCache);
		} catch (_) {
			// Se o navegador não suportar addEventListener aqui, seguimos sem listener.
		}
	}

	// Clique do botão "Limpar saída": apaga o conteúdo do painel #saida.
	if (btnLimparSaida) btnLimparSaida.addEventListener('click', limparSaida);

	// Clique do botão "Comparar": compara a última fala com o texto digitado.
	if (btnComparar) btnComparar.addEventListener('click', compararUltimaFala);

	// Se o usuário apertar Enter dentro do input, também compara.
	if (txtEsperado) {
		txtEsperado.addEventListener('keydown', (e) => {
			if (e.key === 'Enter') compararUltimaFala();
		});
	}

	// Antes de qualquer coisa, popula os DOIS comboboxes usando langspri.js.
	// - Destino (STT/TTS): selLang
	// - Nativo/origem (tradução): selLangNativo
	popularIdiomasNoCombobox(selLang, 'pt-BR');
	popularIdiomasNoCombobox(selLangNativo, 'pt-BR');

	// Atualiza a UI logo no carregamento.
	atualizarTela();

	// =====================================================================
	// RESUMO DIDÁTICO (para estudo rápido)
	// ---------------------------------------------------------------------
	// A ideia desta seção é te dar um “mapa mental” do arquivo.
	// Ela NÃO executa nada; é só documentação.
	//
	// 1) FUNÇÕES (o que fazem e quando usar)
	//
	// normalizarTexto(texto)
	// - Uso: preparar texto para comparar (minúsculo, sem acentos, sem pontuação, espaços normalizados).
	// - É usado antes do Levenshtein e antes de canonizar números.
	//
	// canonizarNumeros(textoNormalizado, lang)
	// - Uso: deixar equivalentes “por extenso” e “numeral” (ex.: "un" -> "1", "um" -> "1").
	// - Entrada: texto JÁ normalizado; escolhe dicionário conforme `lang`.
	//
	// distanciaLevenshtein(a, b)
	// - Uso: medir diferença entre duas strings (número de edições: inserir/apagar/trocar).
	// - Serve para calcular a similaridade e classificar CERTO/MEIO CERTO/ERRADO.
	//
	// compararUltimaFala()
	// - Uso: compara `ultimaTranscricaoFinal` (o que você falou) com `txtEsperado.value` (o que você digitou).
	// - Normaliza + canoniza números + Levenshtein; escreve resultado no status e no painel.
	//
	// setStatus(texto)
	// - Uso: atualizar a faixa/label `#status` no HTML com uma mensagem curta.
	//
	// appendSaida(texto)
	// - Uso: adicionar uma linha no painel `#saida` (log de transcrição, erros, diagnósticos).
	//
	// limparSaida()
	// - Uso: apagar tudo do painel `#saida`.
	//
	// atualizarVozesCache()
	// - Uso: ler `speechSynthesis.getVoices()` e salvar em `vozesCache`.
	// - Chamado no início e também quando o navegador dispara `voiceschanged`.
	//
	// aguardarVozesCarregarem(timeoutMs)
	// - Uso: esperar as vozes TTS carregarem (porque `getVoices()` às vezes começa vazio).
	// - Resolve true/false; evita tentar falar sem voz correta.
	//
	// escolherVozPorIdioma(lang)
	// - Uso: escolher a melhor voz para o idioma do combobox (pt-BR/fr-FR).
	// - Regra: preferir idioma exato e evitar vozes "Multilingual" (podem pronunciar em inglês).
	//
	// diagnosticarVozesTTS(langSelecionado)
	// - Uso: escrever no `#saida` a lista de vozes encontradas para o idioma-base (ex.: fr*).
	// - Ajuda a entender por que o TTS não respeita o idioma.
	//
	// falarTexto(texto)
	// - Uso: fazer o navegador falar via TTS no idioma selecionado.
	// - Passos: cancel() -> cria SpeechSynthesisUtterance -> set lang -> escolhe voz -> speak().
	//
	// atualizarTela()
	// - Uso: atualizar botão e status conforme `sttAtivo` e suporte do navegador.
	//
	// criarRecognitionSePreciso()
	// - Uso: criar e configurar UMA instância de SpeechRecognition, e registrar eventos (onstart/onend/onerror/onresult).
	// - Chamado antes de start.
	//
	// popularIdiomasNoCombobox()
	// - Uso: popular o select `#selLang` automaticamente a partir de `langspri.js`.
	// - Lê `window.LANGS` (nomes) e `window.LANGS_BCP` (códigos BCP-47) e cria as opções.
	//
	// obterIdiomaSelecionado()
	// - Uso: ler `#selLang` (pt-BR/fr-FR). Se não existir, volta para pt-BR.
	//
	// obterIdiomaNativoSelecionado()
	// - Uso: ler `#selLangNativo` (idioma nativo/origem) para a tradução.
	//
	// bcp47ParaBase(bcp47)
	// - Uso: converter BCP-47 (pt-BR) para idioma-base (pt), usado na API de tradução.
	//
	// traduzirNoServidor(texto, origemBcp47, destinoBcp47)
	// - Uso: chamar o endpoint `/api/translate` no server.js para obter o texto traduzido.
	//
	// aoClicarTraduzir()
	// - Uso: handler do botão `#btnTraduzir`.
	// - Traduz o texto do input do idioma nativo para o idioma destino e fala (TTS).
	//
	// setContador(texto)
	// - Uso: escrever texto no elemento `#contador` (contador regressivo do STT).
	//
	// obterTempoSelecionadoSegundos()
	// - Uso: ler `#selTempo`.
	// - Retorna null (sem auto-desligar) ou um número de segundos (>0).
	//
	// limparTemporizadoresSTT()
	// - Uso: cancelar timeout/interval do auto-desligamento e apagar o contador.
	//
	// iniciarTemporizadorSTT(segundos)
	// - Uso: começar a contagem regressiva e agendar o auto-desligamento do STT.
	// - Só é chamado quando o STT realmente começou (no onstart).
	//
	// atualizarContadorRegressivo()
	// - Uso: calcular e mostrar quantos segundos faltam até o auto-desligamento.
	//
	// aoMudarIdiomaDesativarSTT()
	// - Uso: regra de segurança: se mudar idioma com STT ativo, desliga e pede para ativar de novo.
	//
	// window.ativarSTT()
	// - Uso (externo): liga o STT. Faz `recognition.lang = obterIdiomaSelecionado()` e chama `recognition.start()`.
	// - Retorna true/false.
	//
	// window.desativarSTT()
	// - Uso (externo): desliga o STT. Tenta `stop()`, se falhar tenta `abort()`.
	// - Retorna true/false.
	//
	// toggleSTT()
	// - Uso: alterna ativar/desativar. É ligado ao click do botão `#btnSTT`.
	//
	// 2) VARIÁVEIS (para que servem)
	//
	// btnSTT
	// - Referência ao botão `#btnSTT` (Ativar/Desativar STT).
	//
	// statusEl
	// - Referência ao elemento `#status` (mensagem de estado).
	//
	// saidaEl
	// - Referência ao painel `#saida` (log de transcrições/erros/diagnóstico).
	//
	// btnFalar
	// - Botão `#btnFalar` que dispara TTS com a última transcrição.
	//
	// btnLimparSaida
	// - Botão `#btnLimparSaida` que limpa o painel `#saida`.
	//
	// txtEsperado
	// - Input `#txtEsperado` com a palavra/frase que você espera (para comparar).
	//
	// btnComparar
	// - Botão `#btnComparar` que chama `compararUltimaFala()`.
	//
	// btnTraduzir
	// - Botão `#btnTraduzir` que traduz o texto do input e fala no idioma destino.
	//
	// selLang
	// - Select `#selLang` (pt-BR/fr-FR) que define idioma do STT/TTS.
	// - OBS: as opções vêm do `langspri.js` (window.LANGS_BCP).
	//
	// selLangNativo
	// - Select `#selLangNativo` (idioma nativo/origem) usado para tradução.
	//
	// window.LANGS / window.LANGS_BCP
	// - Estruturas globais definidas em `langspri.js`.
	// - `LANGS` = nome amigável; `LANGS_BCP` = código BCP-47 (usado por STT/TTS).
	//
	// selTempo
	// - Select `#selTempo` (null/1/2/3/...) que define se o STT auto-desliga.
	//
	// contadorEl
	// - Elemento `#contador` onde mostramos o contador regressivo na tela.
	//
	// SpeechRecognitionImpl
	// - Implementação do navegador: `SpeechRecognition` (padrão) ou `webkitSpeechRecognition`.
	// - Se for null/undefined, não existe STT via Web Speech nesse browser.
	//
	// recognition
	// - Instância única de SpeechRecognition criada por `criarRecognitionSePreciso()`.
	// - É ela que escuta microfone e gera `onresult`.
	//
	// sttAtivo
	// - Boolean que representa o estado atual do STT (para UI e toggle).
	//
	// ultimaTranscricaoFinal
	// - String com o último resultado FINAL reconhecido (usado pelo botão Falar e pela comparação).
	//
	// sttAutoStopTimeoutId
	// - ID do setTimeout que desliga o STT automaticamente.
	//
	// sttCountdownIntervalId
	// - ID do setInterval que atualiza o contador regressivo.
	//
	// sttAutoStopFimMs
	// - Timestamp (Date.now()) de quando o STT deve ser auto-desligado.
	//
	// sttTempoSolicitadoSegundos
	// - Segundos escolhidos no combobox no momento de ativar (usado no onstart).
	//
	// vozesCache
	// - Array com as vozes TTS disponíveis (`speechSynthesis.getVoices()`).
	// - Usado por `escolherVozPorIdioma()` e pelo diagnóstico.
	// =====================================================================
})();
