// Este arquivo guarda o JavaScript (a parte que dá “vida” à página)

// Guardamos aqui a “conexão” com o microfone quando estiver ligada
let streamDoMicrofone = null;

// Pega o botão pelo id para podermos usar ele no código
const btnMic = document.getElementById('btnMic');

// Pega a área de status pelo id para podermos mostrar mensagens
const status = document.getElementById('status');

// Função pequena para atualizar textos do botão e do status
function atualizarTela() {
	// Se existe um stream, consideramos que o microfone está ligado
	const ligado = streamDoMicrofone !== null;

	// Atualiza o texto do botão conforme o estado
	btnMic.textContent = ligado ? '⏹️ Desativar microfone' : '🎤 Ativar microfone';

	// Atualiza a mensagem para o usuário
	status.textContent = ligado ? 'Status: microfone ligado.' : 'Status: microfone desligado.';
}

// Função que liga o microfone (pede permissão ao navegador)
async function ligarMicrofone() {
	// Se o navegador não tiver a API, avisamos e paramos
	if (!navigator.mediaDevices || !navigator.mediaDevices.getUserMedia) {
		status.textContent = 'Seu navegador não suporta acesso ao microfone (getUserMedia).';
		return;
	}

	try {
		// Pede para o navegador liberar áudio do microfone
		streamDoMicrofone = await navigator.mediaDevices.getUserMedia({ audio: true });
		// Mostra na tela que deu certo
		atualizarTela();
	} catch (err) {
		// Se a pessoa negar permissão ou der erro, mostramos uma mensagem simples
		streamDoMicrofone = null;
		status.textContent = 'Não foi possível ligar o microfone (permissão negada ou erro).';
		console.warn('Erro ao ligar microfone:', err);
	}
}

// Função que desliga o microfone (para as “trilhas” de áudio)
function desligarMicrofone() {
	// Se não tem stream, não há nada para desligar
	if (!streamDoMicrofone) {
		atualizarTela();
		return;
	}

	// Para todas as trilhas (tracks) de áudio, encerrando o microfone
	streamDoMicrofone.getTracks().forEach((t) => t.stop());

	// Limpamos a variável para marcar que está desligado
	streamDoMicrofone = null;

	// Atualiza a tela
	atualizarTela();
}

// Quando a pessoa clicar no botão, alternamos entre ligar e desligar
btnMic.addEventListener('click', async () => {
	// Se está ligado, desligamos
	if (streamDoMicrofone) {
		desligarMicrofone();
		return;
	}

	// Se está desligado, ligamos
	await ligarMicrofone();
});

// Quando a página abre, garante que o texto inicial está correto
atualizarTela();
